﻿<#
.SYNOPSIS
	This script performs the installation or uninstallation of an application(s).
	# LICENSE #
	PowerShell App Deployment Toolkit - Provides a set of functions to perform common application deployment tasks on Windows.
	Copyright (C) 2017 - Sean Lillis, Dan Cunningham, Muhammad Mashwani, Aman Motazedian.
	This program is free software: you can redistribute it and/or modify it under the terms of the GNU Lesser General Public License as published by the Free Software Foundation, either version 3 of the License, or any later version. This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
	You should have received a copy of the GNU Lesser General Public License along with this program. If not, see <http://www.gnu.org/licenses/>.
.DESCRIPTION
	The script is provided as a template to perform an install or uninstall of an application(s).
	The script either performs an "Install" deployment type or an "Uninstall" deployment type.
	The install deployment type is broken down into 3 main sections/phases: Pre-Install, Install, and Post-Install.
	The script dot-sources the AppDeployToolkitMain.ps1 script which contains the logic and functions required to install or uninstall an application.
.PARAMETER DeploymentType
	The type of deployment to perform. Default is: Install.
.PARAMETER DeployMode
	Specifies whether the installation should be run in Interactive, Silent, or NonInteractive mode. Default is: Interactive. Options: Interactive = Shows dialogs, Silent = No dialogs, NonInteractive = Very silent, i.e. no blocking apps. NonInteractive mode is automatically set if it is detected that the process is not user interactive.
.PARAMETER AllowRebootPassThru
	Allows the 3010 return code (requires restart) to be passed back to the parent process (e.g. SCCM) if detected from an installation. If 3010 is passed back to SCCM, a reboot prompt will be triggered.
.PARAMETER TerminalServerMode
	Changes to "user install mode" and back to "user execute mode" for installing/uninstalling applications for Remote Destkop Session Hosts/Citrix servers.
.PARAMETER DisableLogging
	Disables logging to file for the script. Default is: $false.
.EXAMPLE
    powershell.exe -Command "& { & '.\Deploy-Application.ps1' -DeployMode 'Silent'; Exit $LastExitCode }"
.EXAMPLE
    powershell.exe -Command "& { & '.\Deploy-Application.ps1' -AllowRebootPassThru; Exit $LastExitCode }"
.EXAMPLE
    powershell.exe -Command "& { & '.\Deploy-Application.ps1' -DeploymentType 'Uninstall'; Exit $LastExitCode }"
.EXAMPLE
    Deploy-Application.exe -DeploymentType "Install" -DeployMode "Silent"
.NOTES
	Toolkit Exit Code Ranges:
	60000 - 68999: Reserved for built-in exit codes in Deploy-Application.ps1, Deploy-Application.exe, and AppDeployToolkitMain.ps1
	69000 - 69999: Recommended for user customized exit codes in Deploy-Application.ps1
	70000 - 79999: Recommended for user customized exit codes in AppDeployToolkitExtensions.ps1
.LINK
	http://psappdeploytoolkit.com
#>
[CmdletBinding()]
Param (
    [Parameter(Mandatory = $false)]
    [ValidateSet('Install', 'Uninstall', 'Repair')]
    [string]$DeploymentType = 'Install',
    [Parameter(Mandatory = $false)]
    [ValidateSet('Interactive', 'Silent', 'NonInteractive')]
    [string]$DeployMode = 'Interactive',
    [Parameter(Mandatory = $false)]
    [switch]$AllowRebootPassThru = $false,
    [Parameter(Mandatory = $false)]
    [switch]$TerminalServerMode = $false,
    [Parameter(Mandatory = $false)]
    [switch]$DisableLogging = $false
)

Try
{
    ## Set the script execution policy for this process
    Try { Set-ExecutionPolicy -ExecutionPolicy 'ByPass' -Scope 'Process' -Force -ErrorAction 'Stop' }
    Catch { }
    
    ##*===============================================
    ##* VARIABLE DECLARATION
    ##*===============================================
    ## Variables: Application
    [string]$appVendor = 'Cisco'
    [string]$appName = 'AnyConnect_Network_Adapter_Fix'
    [string]$appVersion = '1'
    [string]$appArch = ''
    [string]$appTarget = 'ENT'
    [string]$appLang = 'ENG'
    [string]$appRevision = 'A'
    [string]$appScriptVersion = '1.0.0'
    [string]$appScriptDate = '09/21/2020'
    [string]$appScriptAuthor = 'Lee Emerson'
    [string]$ESDNumber = '5952'
    [string]$ESDRequestor = 'Sivasubramanian Sudalaiyandi'
    [BOOL]$RebootRequired = $false #-$true or $false
    
    ## Keep notes below on what changes were made if $appRevision changes
    # Version B - 
    
    [string]$StartTime = Get-Date -Format g
    [string]$Status = ''
    [string]$FINALMSG = ''
    ##*===============================================
    ## Variables: Install Titles (Only set here to override defaults set by the toolkit)
    #[string]$installName = ''
    [string]$installTitle = 'AnyConnect Network Adapter Fix'
    
    ##* Do not modify section below
    #region DoNotModify
    
    ## Variables: Exit Code
    [int32]$mainExitCode = 0
    
    ## Variables: Script
    [string]$deployAppScriptFriendlyName = 'Deploy Application'
    [version]$deployAppScriptVersion = [version]'3.8.2'
    [string]$deployAppScriptDate = '08/05/2020'
    [hashtable]$deployAppScriptParameters = $psBoundParameters
    
    ## Variables: Environment
    If (Test-Path -LiteralPath 'variable:HostInvocation') { $InvocationInfo = $HostInvocation }
    Else { $InvocationInfo = $MyInvocation }
    [string]$scriptDirectory = Split-Path -Path $InvocationInfo.MyCommand.Definition -Parent
    
    ## Dot source the required App Deploy Toolkit Functions
    Try
    {
        [string]$moduleAppDeployToolkitMain = "$scriptDirectory\AppDeployToolkit\AppDeployToolkitMain.ps1"
        If (-not (Test-Path -LiteralPath $moduleAppDeployToolkitMain -PathType 'Leaf')) { Throw "Module does not exist at the specified location [$moduleAppDeployToolkitMain]." }
        If ($DisableLogging) { . $moduleAppDeployToolkitMain -DisableLogging }
        Else { . $moduleAppDeployToolkitMain }
    }
    Catch
    {
        If ($mainExitCode -eq 0) { [int32]$mainExitCode = 60008 }
        Write-Error -Message "Module [$moduleAppDeployToolkitMain] failed to load: `n$($_.Exception.Message)`n `n$($_.InvocationInfo.PositionMessage)" -ErrorAction 'Continue'
        ## Exit the script, returning the exit code to SCCM
        If (Test-Path -LiteralPath 'variable:HostInvocation') { $script:ExitCode = $mainExitCode; Exit }
        Else { Exit $mainExitCode }
    }
    
    #endregion
    ##* Do not modify section above
    [string]$Attempts = Create-HWCustomRegistryFields
    ##*===============================================
    ##* END VARIABLE DECLARATION
    ##*===============================================
    
    If ($deploymentType -ine 'Uninstall' -and $deploymentType -ine 'Repair')
    {
        ##*===============================================
        ##* PRE-INSTALLATION
        ##*===============================================
        [string]$installPhase = 'Pre-Installation'
        
        
        ##*===============================================
        ##* INSTALLATION
        ##*===============================================
        [string]$installPhase = 'Installation'
        
        ## Handle Zero-Config MSI Installations
        If ($useDefaultMsi)
        {
            [hashtable]$ExecuteDefaultMSISplat = @{ Action = 'Install'; Path = $defaultMsiFile }; If ($defaultMstFile) { $ExecuteDefaultMSISplat.Add('Transform', $defaultMstFile) }
            Execute-MSI @ExecuteDefaultMSISplat; If ($defaultMspFiles) { $defaultMspFiles | ForEach-Object { Execute-MSI -Action 'Patch' -Path $_ } }
        }
        
        Show-InstallationProgress
        
        Write-Log -Message "***************************************************************************************" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        Write-Log -Message "   CHECK IF CISCO ANYCONNECT NETWORK ACCESS MANAGER FILTER IS ENABLED  " -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        Write-Log -Message "***************************************************************************************" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        
        $filterEnabled = Get-NetAdapterBinding -InterfaceDescription Cisco* -ComponentID CSCO_acnamfd
        if ($filterEnabled)
        {
            $filterEnabledName = $filterEnabled.Name
            $filterEnabledDisplayName = $filterEnabled.DisplayName
            $filterEnabledComponentID = $filterEnabled.ComponentID
            $filterEnabled = $filterEnabled.enabled
            
            Write-Log -Message "Name: $filterEnabledName" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
            Write-Log -Message "DisplayName: $filterEnabledDisplayName" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
            Write-Log -Message "ComponentID: $filterEnabledComponentID" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
            Write-Log -Message "Enabled: $filterEnabled" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
            
            
            Write-Log -Message "Cisco AnyConnect Network Access Manager Filter enabled = $filterEnabled" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
            
            if ($filterEnabled -eq $true)
            {
                Write-Log -Message "***************************************************************************************" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
                Write-Log -Message "   RUN DISABLE-NETADAPTERBINDING COMMAND  " -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
                Write-Log -Message "***************************************************************************************" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
                
                Disable-NetAdapterBinding -InterfaceDescription *AnyConnect* -ComponentID CSCO_acnamfd
                
                $filterEnabled = Get-NetAdapterBinding -InterfaceDescription Cisco* -ComponentID CSCO_acnamfd
                
                $filterEnabled = $filterEnabled.enabled
                
                Write-Log -Message "Filter enabled after running command to disable = $filterEnabled" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
            }
        }
        
        Write-Log -Message "***************************************************************************************" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        Write-Log -Message "    VERIFY SUCCESSFUL INSTALL" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        Write-Log -Message "***************************************************************************************" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        
        If (($filterEnabled -eq $false) -or (!$filterEnabled))
        {
            
            Write-Log -Message "Disable was successful" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
            Write-Log -Message "*" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        }
        else
        {
            $FINALMSG = "Disable was unsuccessful"
            $Status = 'Failed'
            Write-Log -Message "$FINALMSG" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName -Severity 3
            
            Write-Log -Message "***************************************************************************************" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
            Write-Log -Message "WRITE TO HW CUSTOM REG KEYS" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
            Write-Log -Message "***************************************************************************************" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
            Update-HWCustomRegistryFields -Status "$Status" -FINALMSG "$FINALMSG" -ExitCode $returnCode
            Show-InstallationPrompt -Message "Disabling of Cisco AnyConnect Network Access Manager Filter failed. Please contact Service Desk for support." -ButtonRightText 'OK' -Icon Information -NoWait
            Exit-Script -ExitCode 1
        }
        
        
        ##*===============================================
        ##* POST-INSTALLATION
        ##*===============================================
        [string]$installPhase = 'Post-Installation'
        
        ## <Perform Post-Installation tasks here>
        
        Write-Log -Message "***************************************************************************************" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        Write-Log -Message "    WRITE TO HW CUSTOM REG KEYS" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        Write-Log -Message "***************************************************************************************" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        Update-HWCustomRegistryFields -Status "Success" -FINALMSG "Successful Installation" -ExitCode $returnCode
        
        Write-Log -Message "***************************************************************************************" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        Write-Log -Message "    RUN HW INVENTORY POLICY" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        Write-Log -Message "***************************************************************************************" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        Invoke-SCCMTask -ScheduleID 'HardwareInventory'
        #Invoke-SCCMTask -ScheduleID 'ApplicationManagerPolicyAction'
        Write-Log -Message "*" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        
        If ($RebootRequired) { Show-InstallationRestartPrompt -NoCountdown }
        #Show-InstallationRestartPrompt -Countdownseconds 28800 -CountdownNoHideSeconds 60
        ## Display a message at the end of the install
        If (-not $useDefaultMsi) { Show-InstallationPrompt -Message "$installTitle was successful." -ButtonRightText 'OK' -Icon Information -NoWait }
    }
    ElseIf ($deploymentType -ieq 'Uninstall')
    {
        ##*===============================================
        ##* PRE-UNINSTALLATION
        ##*===============================================
        [string]$installPhase = 'Pre-Uninstallation'
        
        [string]$installPhase = 'Pre-Uninstallation'
        [string]$configDeferPromptExpiryMessage = $configDeferPromptExpiryMessage.Replace('installation', 'uninstallation')
        [string]$configClosePromptMessage = $configClosePromptMessage.Replace('installation', 'uninstallation')
        [string]$configDiskSpaceMessage = $configDiskSpaceMessage.Replace('installation', 'uninstallation')
        [string]$configRestartPromptMessage = $configRestartPromptMessage.Replace('installation', 'uninstallation')
        [string]$configBlockExecutionMessage = $configBlockExecutionMessage.Replace('installation', 'uninstallation')
        [string]$configDeferPromptWelcomeMessage = $configDeferPromptWelcomeMessage.Replace('IT will be reaching out to you to complete license registration', '')
        [string]$configDeferPromptWelcomeMessage = $configDeferPromptWelcomeMessage.Replace('and can be used after installation.', '')
        [string]$configDeferPromptWelcomeMessage = $configDeferPromptWelcomeMessage.Replace('installed', 'uninstalled')
        [string]$configDeferPromptWelcomeMessage = $configDeferPromptWelcomeMessage.Replace('installation', 'uninstallation')
        
        ## Show Welcome Message, for example to close Internet Explorer type iexplore with a 60 second countdown before automatically closing
        Show-InstallationWelcome -CloseApps 'appname without exe' -CloseAppsCountdown 60
        
        ## Show Progress Message (with the default message)
        Show-InstallationProgress
        
        ## <Perform Pre-Uninstallation tasks here>
        
        
        ##*===============================================
        ##* UNINSTALLATION
        ##*===============================================
        [string]$installPhase = 'Uninstallation'
        
        ## Handle Zero-Config MSI Uninstallations
        If ($useDefaultMsi)
        {
            [hashtable]$ExecuteDefaultMSISplat = @{ Action = 'Uninstall'; Path = $defaultMsiFile }; If ($defaultMstFile) { $ExecuteDefaultMSISplat.Add('Transform', $defaultMstFile) }
            Execute-MSI @ExecuteDefaultMSISplat
        }
        
        Write-Log -Message "***************************************************************************************" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        Write-Log -Message "    UNINSTALL $installTitle" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        Write-Log -Message "***************************************************************************************" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        
        ##  <Example of MSI uninstall>
        ##  <Name as displayed in Programs & Features or in Displayname in uninstall reg key>
        ##  Remove-MSIApplications -Name "Software Name"
        
        
        Write-Log -Message "***************************************************************************************" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        Write-Log -Message "    VERIFY SUCCESSFUL UNINSTALL" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        Write-Log -Message "***************************************************************************************" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        
        ##  <Get one of the methods from the cheat sheet #region UNINSTALL - VERIFY SUCCESSFUL UNINSTALL to verify a successful install>
        
        
        
        ##*===============================================
        ##* POST-UNINSTALLATION
        ##*===============================================
        [string]$installPhase = 'Post-Uninstallation'
        
        Write-Log -Message "***************************************************************************************" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        Write-Log -Message "    WRITE TO HW CUSTOM REG KEYS" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        Write-Log -Message "***************************************************************************************" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        Remove-HWCustomRegistryFields
        Update-HWCustomRegistryFields -Status "Success" -FINALMSG "Successful Uninstall" -ExitCode $returnCode
        
        
        Write-Log -Message "***************************************************************************************" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        Write-Log -Message "    RUN HW INVENTORY POLICY" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        Write-Log -Message "***************************************************************************************" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        Invoke-SCCMTask -ScheduleID 'HardwareInventory'
        #Invoke-SCCMTask -ScheduleID 'ApplicationManagerPolicyAction'
        Write-Log -Message "*" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
        
        If ($RebootRequired) { Show-InstallationRestartPrompt -NoCountdown }
        #Show-InstallationRestartPrompt -Countdownseconds 28800 -CountdownNoHideSeconds 60
        #Show-InstallationPrompt -Message "$installTitle uninstalled successfully"  -ButtonRightText 'OK' -Icon Information -NoWait
        
        
        
    }
    ElseIf ($deploymentType -ieq 'Repair')
    {
        ##*===============================================
        ##* PRE-REPAIR
        ##*===============================================
        [string]$installPhase = 'Pre-Repair'
        
        ## Show Progress Message (with the default message)
        Show-InstallationProgress
        
        ## <Perform Pre-Repair tasks here>
        
        ##*===============================================
        ##* REPAIR
        ##*===============================================
        [string]$installPhase = 'Repair'
        
        ## Handle Zero-Config MSI Repairs
        If ($useDefaultMsi)
        {
            [hashtable]$ExecuteDefaultMSISplat = @{ Action = 'Repair'; Path = $defaultMsiFile; }; If ($defaultMstFile) { $ExecuteDefaultMSISplat.Add('Transform', $defaultMstFile) }
            Execute-MSI @ExecuteDefaultMSISplat
        }
        # <Perform Repair tasks here>
        
        ##*===============================================
        ##* POST-REPAIR
        ##*===============================================
        [string]$installPhase = 'Post-Repair'
        
        ## <Perform Post-Repair tasks here>
        
        
    }
    ##*===============================================
    ##* END SCRIPT BODY
    ##*===============================================
    
    ## Call the Exit-Script function to perform final cleanup operations
    Exit-Script -ExitCode $mainExitCode
}
Catch
{
    
    Write-Log -Message "***************************************************************************************" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
    Write-Log -Message "AN ERROR WAS DETECTED" -LogType 'CMTrace' -Severity 3 -Source $deployAppScriptFriendlyName
    Write-Log -Message "***************************************************************************************" -LogType 'CMTrace' -Source $deployAppScriptFriendlyName
    
    [int32]$mainExitCode = 60001
    [string]$mainErrorMessage = "$(Resolve-Error)"
    Write-Log -Message $mainErrorMessage -Severity 3 -Source $deployAppScriptFriendlyName
    Update-HWCustomRegistryFields -Status "FAILED" -FINALMSG "AN ERROR WAS DETECTED. CHECK SWD LOG" -ExitCode $mainExitCode
    #    if (!$CurrentLoggedOnUserSession)
    #    {
    #        Show-DialogBox -Text $mainErrorMessage -Icon 'Stop'
    #    }
    Exit-Script -ExitCode $mainExitCode
}
